--
-- STEP 1
--

---------------------------------------------------------
-- Create function fn_split_string
---------------------------------------------------------

IF EXISTS (SELECT * FROM sys.objects where object_id = OBJECT_ID(N'[dbo].[fn_split_string]') AND type in (N'TF'))
	DROP FUNCTION dbo.fn_split_string
GO

CREATE FUNCTION [dbo].[fn_split_string]
(
@String		varchar(MAX),
@Delimiter	varchar(1)
)
RETURNS @Table TABLE(I int IDENTITY, Row varchar(128))
AS
BEGIN

/*
Script:			Converts list to table
Author:			r.glen.cooper@gmail.com
Description:	Turns list a,b,c,... into table with identity column
Example:		SELECT * FROM dbo.fn_split_string('a,b,c',',')
*/

WHILE len(@String) > 0
	BEGIN
		INSERT INTO @Table(Row)
		SELECT left(@String, charindex(@Delimiter, @String + @Delimiter) -1) AS Row
		SET @String = stuff(@String, 1, charindex(@Delimiter, @String + @Delimiter), '')
	END
RETURN 
END
GO

---------------------------------------------------------
-- Create procedure sp_VerifyTree
---------------------------------------------------------

IF EXISTS (SELECT * FROM sys.objects where object_id = OBJECT_ID(N'[dbo].[sp_VerifyTree]') AND type in (N'P'))
	DROP PROCEDURE dbo.sp_VerifyTree
GO

CREATE PROCEDURE [dbo].[sp_VerifyTree]
(
@Schema				VARCHAR(128)		-- Schema of table containing tree 
,@Graph				VARCHAR(128)		-- Table containing tree 
,@NodeId			VARCHAR(128)		-- Column name for NodeId in table containing tree
,@Node				VARCHAR(128)		-- Column name for Node in table containing tree
,@ParentId			VARCHAR(128)		-- Column name for ParentId in table containing tree
,@DEBUG				BIT = 0				-- List temp tables during execution
,@DDL				NVARCHAR(MAX) = ''	-- DDL code to execute after successful test (optional)
,@DML				NVARCHAR(MAX) = ''	-- DML code to execute after successful test (optional)
,@ERROR_NUMBER		INT OUTPUT			-- Return error number
,@ERROR_MESSAGE		VARCHAR(MAX) OUTPUT	-- Return error message 
,@ROW_COUNT			INT OUTPUT			-- Return number of rows affected (-1 = not set)
)
AS

BEGIN

/*
Script:		Verify that the table @Graph represents a tree and (optionally) execute custom SQL   
Author:		r.glen.cooper@gmail.com
Notes:		This proc copies the columns @NodeId, @Node, @ParentId in the table @Graph 
			to a temporary table ##Tree which has the column names NodeId, Node, ParentId 
			used by it. It then determines if the temporary table represents a tree. Additionally,
			custom SQL may be executed on either table after a successful test.
			
			The source table @Graph must have a column referencing the "parent" record of any record. 
			The name @ParentId of that column is changed to ParentId in the temporary table, along 
			with changes to @NodeId (key of mode) and @Node (name of node). 
			
			@NodeId and @ParentId must convert to INT.
			@Node must convert to VARCHAR
			
			An undirected graph is called a tree when every two nodes have exactly one 
			undirected path connecting them. If this proc concludes that the table does not 
			represent a tree, the errant nodes are listed to show why.
			
			It does this by attempting to topologically sort the tree, checking for loops in the 
			directed edges defined by the ParentId relationship which would imply that it was not a tree.

			Specifically, it checks that:

			1) Exactly one node has a NULL ParentId
			2) No looping occurs with the ParentId relationship (eg. A -> B -> ... -> A)

			It is easy to prove that the above conditions imply that the nodes and undirected edges 
			of @Graph represent a tree.
*/

-- Declarations
DECLARE @SQL				NVARCHAR(MAX)
DECLARE @Height				INT	
DECLARE @HeightTopNode		INT
DECLARE @FakeNodeId			INT 
DECLARE @NumNodes			INT 
DECLARE @COUNT				INT
DECLARE @I					INT
DECLARE @is_nullable		BIT

-- Declarations for timer
DECLARE @DATE VARCHAR(16) = GETDATE()
DECLARE @DATESECONDS INT =  DATEDIFF(s,@DATE,GETDATE())

-- No counting
SET NOCOUNT ON

-- Assume that @ROW_COUNT is not being computed
SET @ROW_COUNT = -1
-- To compute @ROW_COUNT anywhere for debugging purposes, enter
--	  SET @ROW_COUNT = @@ROWCOUNT
-- immediately after any SELECT, INSERT, DELETE or UPDATE statement
-- and the last value it computes will be returned 

BEGIN TRY 

/*
Set up and populate ##Tree
*/

-- Drop, re-create and populate global temporary table ##Tree with @NodeId, @Node, @ParentId from @Graph	
IF OBJECT_ID(N'tempdb..##Tree') IS NOT NULL DROP TABLE ##Tree

SET @SQL = 'SELECT [' + @NodeId + '] AS NodeId, [' + @Node + '] AS Node, [' + @ParentId + '] AS ParentId INTO ##Tree FROM [' + @Schema + '].[' +@Graph +']'

EXEC dbo.sp_executeSQL @SQL

-- Determine the number of nodes in the tree
SELECT @NumNodes = COUNT(*) FROM ##Tree

-- Check that at least one node exists
IF @NumNodes < 1
	BEGIN
	SET @ERROR_MESSAGE = 'Not a tree: There are no nodes'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- Check if NodeId is unique so it can become a primary key
SELECT @COUNT = COUNT(*) FROM (SELECT NodeId FROM ##Tree GROUP BY NodeId) d   

IF @COUNT < @NumNodes
	BEGIN
	SET @ERROR_MESSAGE = 'The column [' + CAST(@NodeId AS VARCHAR(16)) + '] is not unique and cannot become a primary key'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- Ensure that NodeId is not NULLable so it can become a primary key
-- Note that NodeId is INT so @NodeId must convert to INT
ALTER TABLE ##Tree ALTER COLUMN NodeId INT NOT NULL

-- Add primary key using NodeId
SET @SQL = 'ALTER TABLE ##Tree ADD CONSTRAINT PK_NodeId PRIMARY KEY (NodeId);'

EXEC dbo.sp_executeSQL @SQL

-- Add index using ParentId (doesn't seem to improve performance)
CREATE INDEX NDX_ParentId ON ##Tree (ParentId);

/*
Testing ##Tree
*/

-- Check that exactly one node has a NULL ParentId (abort otherwise)
SELECT @COUNT = COUNT(*) FROM ##Tree WHERE ParentId IS NULL
IF @COUNT <> 1
	BEGIN
	SET @ERROR_MESSAGE = 'Not a tree: There are ' + CAST(@COUNT AS VARCHAR(16)) + ' nodes with a NULL ParentId (must be exactly one)'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- Check that each ParentId refers to an existing node (abort otherwise)
SELECT @COUNT = COUNT(*) FROM ##Tree WHERE ParentId NOT IN (SELECT NodeId FROM ##Tree)
IF @COUNT > 0
	BEGIN
	IF @COUNT > 0 SELECT * FROM ##Tree WHERE ParentId NOT IN (SELECT NodeId FROM ##Tree)
	SET @ERROR_MESSAGE = 'Not a tree: There are ' + CAST(@COUNT AS VARCHAR(16)) + ' nodes with an invalid ParentId'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- Arbitrarily set the Id of a fake node. 
-- This node will be the parent of the top node so that each node
-- has a parent in the loop below (for convenience). 
SET @FakeNodeId = -1  -- Must not be the Id of a real node
 
/*
The loop below sorts the nodes by the longest path to their leaves.
This sorted list is stored in the temporary table #Height. Since the graph is 
finite this loop must terminate within @NumNodes steps unless a loop exists, 
so the incrementing number of steps is monitored to prevent an infinite loop.

First create a preorder table #tblPreorder(A,B) that will list the directed edges 
between the nodes of the tree (ie. node B is the parent of node A). 
Be aware that only the nodes appearing in either column of this table are known 
to the loop below. So when that loop deletes all the records from the preorder table 
representing directed edges from various nodes to the top node, the top node will 
disappear and won't appear in #Height. For that reason a fake edge from the 
top node to the fake node is created. 
*/

--  Drop table #tblPreorder
IF OBJECT_ID(N'tempdb..#tblPreorder') IS NOT NULL DROP TABLE #tblPreorder

-- Add directed edges to #tblPreorder
SELECT NodeId, ParentId
INTO #tblPreorder
FROM ##Tree

-- Count number of edges to be returned by the proc (optional)
SET @ROW_COUNT = @@ROWCOUNT

-- Make the fake node the parent of the top node
UPDATE #tblPreorder
SET ParentId = @FakeNodeId 
WHERE ParentId IS NULL

IF @DEBUG = 1 SELECT '#tblPreorder' AS TempTable, * FROM #tblPreorder ORDER BY NodeId

-- Prepare temporary table #Height
IF OBJECT_ID(N'tempdb..#Height') IS NOT NULL DROP TABLE #Height

CREATE TABLE #Height( 
NodeId	INT,
Height	INT
)

/*
Recursively add nodes in column A of the preorder table to #Height if they don't 
appear in column B of the preorder table, while deleting the rows of the preorder
table in which they appeared (so we don't do this again). Note that the fake node 
guarantees that the top node won't prematurely disappear by earlier deletions, 
preventing it from joining #Height with the correct height. Increment the node height 
for each step, which cannot exceed the number of nodes since we're starting from 0.
If it does, then a loop exists in the tree so it's not really a tree (the remaining edges in
the preorder table will form the loop in the graph that prevents it from being a tree).
*/

IF @DEBUG = 1 PRINT 'Number of nodes in tree =  ' + CAST(@NumNodes AS VARCHAR(16))

SET @Height = 0
WHILE (SELECT COUNT(*) FROM #tblPreorder) > 0
	BEGIN

	-- Add nodes in A to #Height that don't appear in B
	-- These are the nodes in A that have no children
	INSERT INTO #Height(NodeId,Height)
	(
	SELECT NodeId, @Height AS 'Height'
	FROM #tblPreorder
	WHERE
	NodeId NOT IN (
	SELECT ParentId FROM #tblPreorder
	)
	)

	-- Delete rows in #tblPreorder where A doesn't appear in B
	-- These are the directed edges for the nodes in A that have no children
	-- So on the next step, new nodes will now become childless
	DELETE
	FROM #tblPreorder
	WHERE 
	NodeId NOT IN (
	SELECT ParentId FROM #tblPreorder
	)
	 
	-- Increment node height for the next step
	SET @Height = @Height + 1

	-- Abort if height exceeds the number of nodes (ie. loop exists with remaining edges)
	-- List the remaining edges
	-- Subtle point: @Height = @NumNodes works for all trees EXCEPT when each node has at most one child
	IF @Height = @NumNodes + 1
	BEGIN
	SELECT * FROM #tblPreorder
	SELECT @COUNT = COUNT(*) FROM #tblPreorder
	SET @ERROR_MESSAGE = 'Not a tree: One or more loops involving ' + CAST(@COUNT AS VARCHAR(16)) +  ' edges exist in tree.'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

	END
	SET @HeightTopNode = @Height - 1

	IF @DEBUG = 1 
		BEGIN
		PRINT ''
		PRINT 'Height of top node =  ' + CAST(@HeightTopNode AS VARCHAR(16))
		PRINT ''
		END

/*
At this point @Tree is known to be a tree
Now add and populate additional columns to it that are useful for querying
*/

-- Add columns NumDescendants, NumChildren, Height, Depth to ##Tree
ALTER TABLE ##Tree ADD NumChildren		INT NULL
ALTER TABLE ##Tree ADD NumDescendants	INT NULL
ALTER TABLE ##Tree ADD NumLeaves		INT NULL
ALTER TABLE ##Tree ADD Height			INT NULL
ALTER TABLE ##Tree ADD Depth			INT NULL

-- Initialize new columns to 0 (but numLeaves is initialized to 1)
-- Note that NumLeaves is initialized to 1 because NumLeaves is NULL for leaves in ##Tree
UPDATE ##Tree SET NumDescendants = 0, NumChildren = 0, Height = 0, Depth = 0, NumLeaves = 1

-- Update NumChildren, NumDescendants, numLeaves by crawling up the tree from its leaves
-- Note that this computation for nodes at each height employs the values of its children
-- That's why an UPDATE is executed at each level, starting from its leaves (Height = 0)
SET @I = 0
WHILE @I < @HeightTopNode
BEGIN
	SET @I = @I + 1

	UPDATE g 
	SET
	g.NumChildren		=	(SELECT count(*) FROM ##Tree i WHERE i.Parentid = g.Nodeid),
	g.NumDescendants	=	(SELECT sum(NumDescendants) FROM ##Tree h WHERE h.Parentid = g.Nodeid) 
							+ 
							(SELECT count(*) FROM ##Tree k WHERE k.Parentid = g.Nodeid),
	g.NumLeaves			=	(SELECT sum(NumLeaves) FROM ##Tree j WHERE j.Parentid = g.Nodeid)
	FROM ##Tree g
	WHERE g.NodeId IN (SELECT Nodeid FROM #Height WHERE Height = @I)
END

-- Update Height
UPDATE t
SET t.Height = h.Height
FROM ##Tree t
INNER JOIN #Height h
ON t.NodeId = h.NodeId 

IF @DEBUG = 1
	SELECT '#Height'  AS TempTable, * FROM #Height ORDER BY Height DESC

-- Update Depth 
-- Crawl down the tree from its top node
IF OBJECT_ID(N'tempdb..#Depth') IS NOT NULL DROP TABLE #Depth

;WITH cte (NodeId, Depth) AS (
    SELECT NodeId, 0 AS Depth	-- Add top node to cte
    FROM ##Tree  
    WHERE NodeId = (SELECT TOP 1 NodeId WHERE ParentId IS NULL)

    UNION ALL 

    SELECT g.NodeId, cte.Depth + 1 AS Depth	-- Add children of nodes currently in cte 
    FROM ##Tree g
    INNER JOIN cte ON cte.NodeId = g.ParentId 
) 
SELECT NodeId, Depth INTO #Depth FROM cte	-- Collect all depths from cte into #Depth

IF @DEBUG = 1
	SELECT '#Depth'  AS TempTable, * FROM #Depth ORDER BY Depth DESC

UPDATE g
SET g.Depth = h.Depth
FROM
##Tree g INNER JOIN #Depth h
ON g.NodeId = h.NodeId 

IF @DEBUG = 1
	SELECT '##Tree'  AS TempTable, * FROM ##Tree ORDER BY NodeId

-- Execute DDL in @DDL
-- Do this first so @DML sees changes to database structure
IF trim(@DDL) <> '' 
	BEGIN
	IF @DEBUG = 1
		BEGIN
		PRINT '@DDL'
		PRINT '----'
		PRINT @DDL
		PRINT ''
		END
	EXEC dbo.sp_executeSQL @DDL
	END

-- Execute DML in @DML
IF trim(@DML) <> '' 
	BEGIN
	IF @DEBUG = 1
		BEGIN
		PRINT '@DML'
		PRINT '----'
		PRINT @DML
		END
	EXEC dbo.sp_executeSQL @DML
	END

END TRY

/*
Error
*/

BEGIN CATCH
SET @ERROR_MESSAGE = ERROR_MESSAGE()
SET @ERROR_NUMBER = ERROR_NUMBER()
-- Drop global temp table ##Tree
IF OBJECT_ID(N'tempdb..##Tree') IS NOT NULL DROP TABLE ##Tree
RETURN @ERROR_NUMBER
END CATCH

/*
Normal return
*/

-- Drop global temp table ##Tree if @DEBUG = 0 
IF @DEBUG = 0 IF OBJECT_ID(N'tempdb..##Tree') IS NOT NULL DROP TABLE ##Tree

SET @ERROR_MESSAGE = ''
SET @ERROR_NUMBER = 0
RETURN @ERROR_NUMBER

END
GO

---------------------------------------------------------
-- Create procedure sp_HorizontalTree
---------------------------------------------------------

IF EXISTS (SELECT * FROM sys.objects where object_id = OBJECT_ID(N'[dbo].[sp_HorizontalTree]') AND type in (N'P'))
	DROP PROCEDURE dbo.sp_HorizontalTree
GO

CREATE PROCEDURE [dbo].[sp_HorizontalTree]
(
@Server			VARCHAR(128)	= ''	-- Server
,@Database		VARCHAR(128)	= ''	-- Database
,@Schema		VARCHAR(128)	= ''	-- Schema
,@Table			VARCHAR(128)	= ''	-- Table
,@Columns		VARCHAR(MAX)	= ''	-- List of columns used (all others are ignored)
,@FirstNodes	VARCHAR(MAX)	= ''	-- List of acceptable values for first column 	
,@DEBUG			INT				= 0		-- 0 = no debugging, 1 = show all temp tables
,@ERROR_NUMBER	INT OUTPUT				-- Return error number
,@ERROR_MESSAGE	VARCHAR(MAX) OUTPUT		-- Return error message 
,@ROW_COUNT		INT OUTPUT				-- Return number of rows affected (-1 = not set)
)
AS

BEGIN

/*
Script:		Convert any table to a tree using a selected list of dependant columns
Author:		r.glen.cooper@gmail.com
Required:	dbo.fn_split_string, dbo.sp_VerifyTree
Note:		All rows from @Table, where at least one of @Columns is NULL, are ignored
Note:		Column names in @Columns may not contain blanks
Note:		If @DEBUG = 1 then the global temporary tables created by the proc will appear in the output
			The names of the global temporary tables are dynamically set by the proc 
			This is done by appending a random 5-character suffix to each of them
			Set @DEBUG = 1 to view the suffix in the Messages pane of SSMS
			All global temporary tables created by this session are dropped if @DEBUG = 0
			They are also dropped when its own session is dropped
Note:		If your remote table has a geography data type you may get the following error:
			Objects exposing columns with CLR types are not allowed in distributed queries 
			Instead, copy the table into your local database (use OPENQUERY() if necessary)
Comment:	A column is dependant on another if the meaning of one depends on the other
			For example, Fiscal Quarter is dependent on Fiscal Year in an accounting table
			This proc converts dependant columns into a tree where each node is dependant on its parent
Algorithm:	The proc first creates a copy of @Server.@Database.@Schema.@Table in ##Table on which filtering is performed
			Then it creates ##TempStage to store just the columns @Columns of ##Table, along with lookup columns
			It also creates a lookup table for each column in @Columns 
			The lookup columns in ##TempStage reference the observed values in ##Table stored in the lookup tables 
			They serve as primary keys for the nodes in the tree
			They also serve as foreign keys for their adjacent nodes 
			The nodes for these columns are inserted into the tree in the order they appear in @Columns
			Each insertion for columns 2,3,4, ... first creates ##TempTable with IDENTITY 
			This ensures that unique primary key values are inserted into the tree for each column
			The ParentId for each node uses the NodeId of the preceding column 
			After each insertion of a column, the NodeId for that column is updated in ##TempStage 
			That way, the insertion of the next column has the correct values for ParentId
			The first column must be character based for its character-based joins
			COLLATE DATABASE_DEFAULT is used in the joins involving the first column of @Columns 
			That's because @Server.@Database.@Schema.@Table may have a different collation than the host database
*/

-- Declarations
DECLARE @SQL				NVARCHAR(MAX)
DECLARE @I					INT
DECLARE @NextNodeId			INT
DECLARE @NumColumns			INT
DECLARE @NumFirstNodes		INT
DECLARE @ColumnId			INT
DECLARE @FirstColumn		NVARCHAR(128)
DECLARE @NextColumn			VARCHAR(128)
DECLARE @NextValue			VARCHAR(128)
DECLARE @NumRec				INT
DECLARE @ColumnsDelimited	NVARCHAR(MAX)
DECLARE @INPUT				NVARCHAR(128)
DECLARE @OUTPUT				INT
DECLARE @OUTPUT1			NVARCHAR(MAX)
DECLARE @OUTPUT2			INT 
DECLARE @ParmDefinition		NVARCHAR(MAX)
DECLARE @LastColumn			NVARCHAR(128)
DECLARE @CurrentColumn		NVARCHAR(128)
DECLARE @Height				INT
DECLARE @MaxHeight			INT
DECLARE @ColumnsNULL		NVARCHAR(MAX)
DECLARE @Count				INT
DECLARE @Count1				INT
DECLARE @Count2				INT
DECLARE @Suffix				NVARCHAR(16)

-- Set random suffix for global temporary tables
SET @Suffix = FLOOR(100000 *rand()) 

-- Declarations for global temporary tables
-- Note that there are also global temporary tables for each column in @Columns
-- The names of these columns are unknown when proc begins
DECLARE @##Table		NVARCHAR(128) = '##Table'		+ @Suffix
DECLARE @##HorizTree	NVARCHAR(128) = '##HorizTree'	+ @Suffix
DECLARE @##FirstNodes	NVARCHAR(128) = '##FirstNodes'	+ @Suffix
DECLARE @##TopNode		NVARCHAR(128) = '##TopNode'		+ @Suffix
DECLARE @##TempStage	NVARCHAR(128) = '##TempStage'	+ @Suffix
DECLARE @##TempTable	NVARCHAR(128) = '##TempTable'	+ @Suffix

-- No counting
SET NOCOUNT ON

-- Assume that @ROW_COUNT is not being computed in this script:
SET @ROW_COUNT = -1
-- You may compute @ROW_COUNT, for example, by entering:
--	  SET @ROW_COUNT = @@ROWCOUNT 
-- immediately after a SELECT, INSERT, DELETE, UPDATE statement
-- Only the last computation is returned

/*
Error check input data before execution begins
As well, compute @FirstColumn, @NumColumns, @ColumnsDelimited, @NumFirstNodes
*/

--
-- Check that @Server exists
--

IF  @Server IS NULL OR TRIM(@Server) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No server entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

IF NOT EXISTS(SELECT name FROM sys.servers WHERE name = TRIM(@Server))
	BEGIN
	SET @ERROR_MESSAGE = 'Server [' + @Server + '] does not exist or you do not have permission to access it'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

--
-- Check that @Server.@Database exists
--

IF  @Database IS NULL OR TRIM(@Database) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No database entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

SET @SQL = '
IF NOT EXISTS(SELECT name FROM [' + @Server + '].[Master].sys.databases WHERE name = ''' + @Database + ''')
	BEGIN
	SET @OUTPUT1 = ''Database [' + @Server + '].[' + @Database + '] does not exist or you do not have permission to access it''
	SET @OUTPUT2 = 999
	END
ELSE
	BEGIN
	SET @OUTPUT1 = ''''
	SET @OUTPUT2 = 0
	END
'
SET @ParmDefinition = N'@OUTPUT1 NVARCHAR(MAX) OUTPUT, @OUTPUT2 INT OUTPUT'
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @OUTPUT1 = @OUTPUT1 OUTPUT, @OUTPUT2 = @OUTPUT2 OUTPUT

SET @ERROR_MESSAGE = @OUTPUT1	
SET @ERROR_NUMBER = @OUTPUT2
IF @ERROR_NUMBER <> 0
	RETURN @ERROR_NUMBER

-- 
-- Check that @Server.@Database.@Schema exists
--

IF  @Schema IS NULL OR TRIM(@Schema) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No Schema entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

SET @SQL = '
SELECT * INTO #t FROM [' + @Server + '].[' + @Database + '].sys.schemas WHERE name = ''' + @Schema + '''
'
EXEC dbo.sp_executeSQL @SQL

IF @@ROWCOUNT = 0 
	BEGIN
	SET @ERROR_MESSAGE = 'Schema [' + @Schema + '] does not exist in [' + @Server + '].[' + @Database + ']'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- 
-- Check that @Server.@Database.@Schema.@Table exists
--

IF  @Table IS NULL OR TRIM(@Table) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No Table entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

SET @SQL = '
SELECT * INTO #t FROM [' + @Server + '].[' + @Database + '].sys.tables WHERE name = ''' + @Table + '''
'
EXEC dbo.sp_executeSQL @SQL

IF @@ROWCOUNT = 0 
	BEGIN
	SET @ERROR_MESSAGE = 'Table [' + @Table + '] does not exist in [' + @Server + '].[' + @Database + '].' 
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

--
-- Check that at least one column in @Columns exist
--

IF  @Columns IS NULL OR TRIM(@Columns) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No Columns entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- 
-- Check that first column of @Columns is character based 
-- This is because it will be joined with temporary tables in host database
-- @Server.@Database.@Schema.@Table may have different collation from host database 
-- COLLATE DATABASE_DEFAULT is used with its joins to avoid this

-- Get first column
SELECT @FirstColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = 1

SELECT
@Count = COUNT(DATA_TYPE)
FROM
INFORMATION_SCHEMA.COLUMNS
WHERE
TABLE_SCHEMA = @Schema AND
TABLE_NAME = @Table 
AND
COLUMN_NAME = @FirstColumn 
AND
LOWER(DATA_TYPE) NOT IN ('char', 'varchar', 'text', 'nchar', 'nvarchar', 'ntext')

IF @Count > 0
	BEGIN
	SET @ERROR_MESSAGE = 'The first column in @Columns (' + @FirstColumn + ') must be one of char, varchar, text, nchar, nvarchar, ntext'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- 
-- Check that names of all columns in @Columns contain no blanks, are unique and exist in  @Server.@Schema.@Table
-- If necessary, you may temporarily remove blanks from column names
--

-- Get number of columns
SELECT @NumColumns = COUNT(*) FROM [dbo].[fn_split_string](@Columns,',')

SET @I = 0
WHILE @I < @NumColumns
	BEGIN
	SET @I = @I + 1
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @I 
	IF CHARINDEX(' ', @NextColumn) > 0 
		BEGIN
		SET @ERROR_MESSAGE = 'Column [' + @NextColumn + '] cannot contain blanks'
		SET @ERROR_NUMBER = 999
		RETURN @ERROR_NUMBER
		END
	END

SELECT @Count1 = COUNT(TRIM(Row)) FROM [dbo].[fn_split_string](@Columns,',')
SELECT @Count2 = COUNT(*) FROM (SELECT TRIM(Row) AS Row FROM [dbo].[fn_split_string](@Columns,',') GROUP BY TRIM(Row)) d 
IF @Count2 < @Count1
	BEGIN
	SET @ERROR_MESSAGE = '@Columns has duplicate values'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

SET @I = 0
WHILE @I < @NumColumns
	BEGIN

	SET @I = @I + 1
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @I 
	SET @SQL = '
	SELECT COLUMN_NAME INTO #t FROM [' + @Server + '].[' + @Database + '].INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ''' + @Schema + ''' AND TABLE_NAME =  ''' + @Table + ''' AND COLUMN_NAME = ''' + @NextColumn + ''''
	EXEC dbo.sp_executesql @SQL

	IF @@ROWCOUNT = 0 
	BEGIN
		SET @ERROR_MESSAGE = 'Column [' + @NextColumn + '] does not exist in table [' + @Server + '].[' + @Database + '].[' + @Schema + '].[' + @Table + ']'
		SET @ERROR_NUMBER = 999
		RETURN @ERROR_NUMBER
	END

	END

-- 
-- Check that all columns in @Columns are not NULL for at least one record in @Server.@Database.@Schema.@Table
-- Otherwise no data will be processed since records containing NULLs are ignored
--

-- Express @Columns with delimiters
SET @ColumnsDelimited = ''
SELECT @ColumnsDelimited =  @ColumnsDelimited + '[' + TRIM(Row) + '],' FROM [dbo].[fn_split_string](@Columns,',') ORDER BY I
SET @ColumnsDelimited = SUBSTRING(@ColumnsDelimited,1,len(@ColumnsDelimited) - 1)

SET @SQL = @ColumnsDelimited 
SET @SQL = replace(@SQL,',',' IS NOT NULL AND ')
SET @SQL = @SQL + ' IS NOT NULL'

SET @SQL = N'SELECT COUNT(*) INTO #t FROM [' + @Server + '].[' + @Database + '].[' + @Schema + '].[' + @Table + '] WHERE ' + @SQL

IF @@ROWCOUNT = 0 
	BEGIN
	SET @ERROR_MESSAGE = 'No records in [' + @Server + '].[' + @Database + '].[' + @Schema + '].[' + @Table + '] have non-NULL values for all columns in @Columns'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

--
-- Check that all values in @FirstNodes are unique and occur at least once in first column of @Columns
--

IF  @FirstNodes IS NULL OR TRIM(@FirstNodes) = ''
	BEGIN
	SET @ERROR_MESSAGE = 'No @FirstNodes entered'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

SELECT @Count1 = COUNT(TRIM(Row)) FROM [dbo].[fn_split_string](@FirstNodes,',')
SELECT @Count2 = COUNT(*) FROM (SELECT TRIM(Row) AS Row FROM [dbo].[fn_split_string](@FirstNodes,',') GROUP BY TRIM(Row)) d 
IF @Count2 < @Count1
	BEGIN
	SET @ERROR_MESSAGE = '@FirstNodes has duplicate values'
	SET @ERROR_NUMBER = 999
	RETURN @ERROR_NUMBER
	END

-- Get number of first nodes
SELECT @NumFirstNodes = COUNT(*) FROM [dbo].[fn_split_string](@FirstNodes,',')

SET @I = 0
WHILE @I < @NumFirstNodes 
	BEGIN
	SET @I = @I + 1
	SELECT @NextValue = TRIM(Row) FROM [dbo].[fn_split_string](@FirstNodes,',') WHERE I = @I 
	SET @SQL = 'SELECT @OUTPUT = COUNT(' + @FirstColumn + ') FROM [' + @Server + '].[' + @Database + '].[' + @Schema + '].[' + @Table + '] WHERE ' + @FirstColumn + ' = ''' + @NextValue + ''''
	SET @ParmDefinition = N'@OUTPUT INT OUTPUT'
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @OUTPUT = @OUTPUT OUTPUT;
	IF @OUTPUT = 0
		BEGIN
		SET @ERROR_MESSAGE = 'The @FirstNodes value ''' + @NextValue + ''' does not appear in column [' + @FirstColumn + '] of table  [' + @Server + '].[' + @Database + '].[' + @Schema + '].[' + @Table + ']'
		SET @ERROR_NUMBER = 999
		RETURN @ERROR_NUMBER
		END
	END

/*
Execution
*/

BEGIN TRY

-- No counting
SET NOCOUNT ON

-- Insert @Server.@Database.@Schema.@Table into global temporary table ##Table
-- This is the source of data that will be filtered by @FirstNodes values
-- These values will also be used to define the children of the top node
-- This operation will fail for remote tables with the geography data type

SET @SQL = '
IF OBJECT_ID(N''tempdb..' + @##Table + ''') IS NOT NULL DROP TABLE ' + @##Table + '
SELECT ' + @ColumnsDelimited + ' INTO ' +  @##Table + ' FROM [' + @Server + '].[' + @Database + '].[' + + @Schema + '].[' + @Table + '] 
'
EXEC dbo.sp_executesql @SQL

-- Delete rows from ##Table where a NULL column exists
-- NULLs are not allowed in data source
SET @ColumnsNULL = replace(@Columns,',',' IS NULL OR ')
SET @ColumnsNULL = @ColumnsNULL + ' IS NULL'
SET @SQL = 'DELETE FROM ' + @##Table + ' WHERE ' + @ColumnsNULL 
EXEC dbo.sp_executesql @SQL

-- Drop ##HorizTree and re-build 
-- It will eventually be passed to dbo.sp_VerifyTree which verifies and builds the tree
SET @SQL = '
IF OBJECT_ID(N''tempdb..' + @##HorizTree + ''') IS NOT NULL DROP TABLE ' + @##HorizTree + '

CREATE TABLE ' + @##HorizTree + '(
	[NodeId]			[int]			NOT NULL,
	[Node]				[varchar](128)	NOT NULL,
	[ParentId]			[int]			NULL,
	[NumChildren]		[int]			NULL,
	[NumDescendants]	[int]			NULL,
	[NumLeaves]			[int]			NULL,
	[Height]			[int]			NULL,
	[Depth]				[int]			NULL
) ON [PRIMARY]
'
EXEC dbo.sp_executesql @SQL

-- Store the acceptable list of values of first column in ##FirstNodes
-- This global temporary table is used to filter ##Table in what follows
-- It has nothing to do with building the tree 

SET @SQL = '
IF OBJECT_ID(N''tempdb..' + @##FirstNodes + ''') IS NOT NULL DROP TABLE ' + @##FirstNodes + '
SELECT I AS I, TRIM(Row) AS Row INTO ' + @##FirstNodes + ' FROM [dbo].[fn_split_string](''' + @FirstNodes + ''','','') 
'
EXEC dbo.sp_executesql @SQL

/*
Build lookup table for top node 
Note that there's only one record and NodeId = 1
This key won't be changed in tree 
*/

SET @SQL = '
IF OBJECT_ID(N''tempdb..' + @##TopNode + ''') IS NOT NULL DROP TABLE ' + @##TopNode + '
CREATE TABLE ' + @##TopNode + '(NodeId INT IDENTITY(1,1), Node VARCHAR(128))
INSERT INTO ' + @##TopNode + '(Node) 
SELECT DISTINCT ''TOP''
'
EXEC dbo.sp_executesql @SQL

IF @DEBUG = 1
BEGIN
	SET @SQL = 'SELECT ''' + @##TopNode + ''' AS ''Column 0'', * FROM ' + @##TopNode 
	EXEC dbo.sp_executesql @SQL
END

/*
Build lookup table of observed values of first column in source data 
Note that NodeId = 2,3,...
These keys won't be changed in tree (unlike the remaining columns)
*/

SET @SQL = '
IF OBJECT_ID(N''tempdb..##' + @FirstColumn + @Suffix + ''') IS NOT NULL DROP TABLE ##' + @FirstColumn + @Suffix + '
CREATE TABLE ##' + @FirstColumn + @Suffix + '(NodeId INT IDENTITY(2,1), Node VARCHAR(128))
INSERT INTO ##' + @FirstColumn + @Suffix + '(Node) 
SELECT DISTINCT ' + @FirstColumn + '
FROM ' + @##Table + ' WHERE ' + @FirstColumn + ' COLLATE DATABASE_DEFAULT IN (SELECT Row COLLATE DATABASE_DEFAULT FROM ' + @##FirstNodes + ')
'
EXEC dbo.sp_executesql @SQL

IF @DEBUG = 1
BEGIN
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = 1
	SET @SQL = 'SELECT ''##' + @NextColumn + @Suffix + ''' AS ''Column 1'', * FROM ##' + @NextColumn + @Suffix
	EXEC dbo.sp_executesql @SQL
END

/*
Build lookup tables for observed values of remaining columns in source data
*/

SET @ColumnId = 1

WHILE @ColumnId < @NumColumns
	BEGIN
	-- Get next column in @Columns
	SET @ColumnId = @ColumnId + 1
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @ColumnId

	-- Build lookup table for it
	-- Note that NodeId = 1,2,3,...
	-- These values will be changed in tree to avoid conflict with NodeId values inserted earlier
	-- At this point the proc has no idea what NodeIds have already been generated
	SET @SQL = '
	IF OBJECT_ID(N''tempdb..##' + @NextColumn + @Suffix + ''') IS NOT NULL DROP TABLE ##' + @NextColumn + @Suffix + '
	CREATE TABLE ##' + @NextColumn + @Suffix + '(NodeId INT IDENTITY(1,1), Node VARCHAR(128))
	INSERT INTO ##' + @NextColumn + @Suffix + '(Node) 
	SELECT DISTINCT ' + @NextColumn + '
	FROM ' + @##Table + ' WHERE ' + @FirstColumn + ' COLLATE DATABASE_DEFAULT IN (SELECT Row COLLATE DATABASE_DEFAULT FROM ' + @##FirstNodes + ')
	'
	EXEC dbo.sp_executesql @SQL

	IF @DEBUG = 1
	BEGIN
		SET @SQL = 'SELECT ''##' + @NextColumn + @Suffix + ''' AS ''Column ' + CAST(@ColumnId AS VARCHAR(16)) + ''', * FROM ##' + @NextColumn + @Suffix
		EXEC dbo.sp_executesql @SQL
	END
END

/*
Build global temporary table ##TempStage (with newly-created lookup keys) to store ##Table
These lookup keys, which will be updated as tree-building progresses, will reference the lookup tables for @Columns 
*/

-- Snippet to build table's key and first column with its own lookup key
SET @SQL = 'IF OBJECT_ID(N''tempdb..' + @##TempStage + ''') IS NOT NULL DROP TABLE ' + @##TempStage + CHAR(13) + CHAR(10) + 'CREATE TABLE ' + @##TempStage + '(' + CHAR(13) + CHAR(10)
SET @SQL = @SQL + 'I INT IDENTITY(1,1),' + CHAR(13) + CHAR(10)
SET @SQL = @SQL + 'Top_Node	VARCHAR(128),' + CHAR(13) + CHAR(10)
SET @SQL = @SQL + 'Top_Node_Id INT,' 

SET @I = 0

WHILE @I < @NumColumns
	BEGIN
	SET @I = @I + 1
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE @I = I
	-- Snippet to build next column with its own lookup key
	SET @SQL = @SQL + CHAR(13) + CHAR(10) + @NextColumn + ' VARCHAR(128),' +  CHAR(13) + CHAR(10) + @NextColumn + '_Id INT'               + ','
	END

SET @SQL = substring(@SQL, 1, len(@SQL) - 1)

SET @SQL = @SQL + CHAR(13) + CHAR(10) + ')'

EXEC dbo.sp_executesql @SQL

-- Insert ##Table into ##TempStage
-- After this, lookup keys are computed before building tree
SET @SQL = '
INSERT INTO ' + @##TempStage  + '(' + @ColumnsDelimited + ')
SELECT ' + @ColumnsDelimited + ' FROM ' + @##Table + ' WHERE ' + @FirstColumn + ' COLLATE DATABASE_DEFAULT IN (SELECT Row COLLATE DATABASE_DEFAULT FROM ' + @##FirstNodes + ')
'
EXEC dbo.sp_executesql @SQL

IF @DEBUG = 1 
	BEGIN
	SET @SQL = 'SELECT * FROM ' + @##TempStage
	EXEC dbo.sp_executesql @SQL
	END

/*
Compute the lookup keys in ##TempStage which reference the lookup tables
These keys will be used as a primary key when building tree from the top down
However, they will be tranformed as tree-building progresses to avoid clashes
Such transformations will be updated in ##TempStage 
That way, each column's foreign key values in the tree will be correct
That's because each column's foreign key is the lookup key of its preceding column
That's why insertions are performed in the same order they appear in @Columns
*/

-- Top_Node
SET @SQL = '
UPDATE ' + @##TempStage + ' SET Top_Node = ''Top_Node'', Top_Node_Id = 1
'
EXEC dbo.sp_executesql @SQL

-- Remaining nodes
SET @ColumnId = 0
WHILE @ColumnId < @NumColumns
	BEGIN
	SET @ColumnId = @ColumnId + 1
	SELECT @NextColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @ColumnId
	SET @SQL = '
	UPDATE D 
	SET D.' + @NextColumn + '_Id = X.NodeId
	FROM ' +
	@##TempStage + ' D 
	INNER JOIN
	##' + @NextColumn + @Suffix + ' X
	ON D.' + @NextColumn + ' = X.[Node]
	'
	EXEC dbo.sp_executesql @SQL
	END

IF @DEBUG = 1
	BEGIN 
	SET @SQL = 'SELECT * FROM ' + @##TempStage
	EXEC dbo.sp_executesql @SQL
	END

/*
Add nodes to tree starting at the top node and working down
*/

--
-- Add top node
--

-- Insert top node to target table 
SET @SQL = '
INSERT INTO ' + @##HorizTree + '(NodeId, Node, ParentId) SELECT DISTINCT Top_Node_Id AS NodeId, ''Top_Node'' AS Node, NULL AS ParentId FROM ' + @##TempStage
EXEC dbo.sp_executeSQL @SQL

---
-- Add nodes for first column
--

-- Insert nodes for first column into target table
-- Note that the lookup keys for the top node and first node don't need to be transformed
SET @SQL = '
INSERT INTO ' + @##HorizTree + '(NodeId, Node, ParentId) SELECT DISTINCT ' + @FirstColumn + '_Id AS NodeId, ' + @FirstColumn + ' AS Node, Top_Node_Id AS ParentId FROM ' + @##TempStage
EXEC dbo.sp_executesql @SQL

-- Get next available NodeId for next column
SET @SQL = N'SELECT @OUTPUT = MAX(NodeId) FROM ' + @##HorizTree 
SET @ParmDefinition = N'@OUTPUT INT OUTPUT'
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @OUTPUT = @OUTPUT OUTPUT;
SET @NextNodeId = @OUTPUT
SET @NextNodeId = @NextNodeId + 1

--
-- Add nodes for remaining columns
--

SET @ColumnId = 1

WHILE @ColumnId < @NumColumns
	BEGIN
	SET @ColumnId = @ColumnId + 1

	-- First add distinct records in ##TempStage for current column to global temporary table ##TempTable with IDENTITY column 
	-- That way, IDENTITY generates new primary key values for tree starting at next available value
	-- Note that @LastColumn + '_Id has already been updated in previous loop
	SELECT @CurrentColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @ColumnId
	SELECT @LastColumn = TRIM(Row) FROM [dbo].[fn_split_string](@Columns,',') WHERE I = @ColumnId - 1

	SET @SQL = '
	IF OBJECT_ID(N''tempdb..' + @##TempTable + ''') IS NOT NULL DROP TABLE ' + @##TempTable + '
	CREATE TABLE ' + @##TempTable + '(I INT IDENTITY(' + CAST(@NextNodeId AS VARCHAR(16)) + ',1), Node VARCHAR(128), ParentId INT)
	INSERT INTO ' + @##TempTable + '(Node, ParentId) SELECT DISTINCT ' + @CurrentColumn + ' AS Node, ' + @LastColumn + '_Id AS ParentId FROM ' +  @##TempStage
	EXEC dbo.sp_executesql @SQL

	-- Insert ##TempTable to tree with newly-generated keys for NodeId
	-- Then compute the next available NodeId for the next insertion
	SET @SQL = '
	INSERT INTO ' + @##HorizTree + '(NodeId, Node, ParentId) SELECT I, Node, ParentId FROM ' + @##TempTable + '
	SELECT @OUTPUT = MAX(NodeId) FROM ' + @##HorizTree 
	SET @ParmDefinition = N'@OUTPUT INT OUTPUT'
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @OUTPUT = @OUTPUT OUTPUT;
	SET @NextNodeId = @OUTPUT

	SET @NextNodeId = @NextNodeId + 1

	-- Re-compute lookup key in ##TempStage for current column
	-- That way, insertions for the next column will have the correct ParentId in ##TempStage
	SET @SQL = '
	UPDATE dd
	SET dd.' + @CurrentColumn + '_Id = d.NodeId
	FROM ' +
	@##HorizTree + ' d
	INNER JOIN ' +
	@##TempStage + ' dd
	ON d.Node COLLATE DATABASE_DEFAULT = dd.' + @CurrentColumn + ' COLLATE DATABASE_DEFAULT
	AND
	d.ParentId = dd.' + @LastColumn + '_Id
	'
	EXEC dbo.sp_executesql @SQL
	END

IF @DEBUG = 1 
	BEGIN
	SET @SQL = 'SELECT * FROM ' + @##TempStage
	EXEC dbo.sp_executesql @SQL
	END
/*
Update numChildren, numDescendants, numLeaves, Height, Depth
*/

sp_VerifyTree: -- Debugging label used to skip processing during development (eg. GOTO sp_VerifyTree)

SET @SQL = '
DECLARE @ERROR_NUMBER	INT 			
DECLARE @ERROR_MESSAGE	VARCHAR(MAX) 
DECLARE @ROW_COUNT		INT 

EXEC @ERROR_NUMBER	=
dbo.sp_VerifyTree
@Schema			= ''dbo'',					-- Not used for global temp tables (just use dbo to pass error checking) 
@Graph			= ''' + @##HorizTree + ''',	-- table containing tree
@NodeId			= ''NodeId'',				-- column name for NodeId in table containing tree
@Node			= ''Node'',					-- column name for Node in table containing tree
@ParentId		= ''ParentId'',				-- column name for ParentId in table containing tree
@DDL			= '''',						-- DDL code to execute first (optional)
@DML			= ''
-- Update columns in source table
UPDATE g 
SET 
g.NumChildren = t.NumChildren,
g.NumDescendants  = t.NumDescendants,
g.NumLeaves = t.NumLeaves,
g.Height = t.Height,
g.Depth = t.Depth
FROM ' + 
@##HorizTree + ' g INNER JOIN ##Tree t
ON g.NodeId = t.NodeId
'',											-- DML code to execute next (optional)
@ERROR_NUMBER	= @ERROR_NUMBER		OUTPUT,
@ERROR_MESSAGE	= @ERROR_MESSAGE	OUTPUT,
@ROW_COUNT		= @ROW_COUNT		OUTPUT

IF @ERROR_NUMBER <> 0 SELECT @ERROR_NUMBER AS ERROR_NUMBER, @ERROR_MESSAGE AS ERROR_MESSAGE, @ROW_COUNT AS ROW_COUNT -- Optional
'
EXEC dbo.sp_executesql @SQL

/*
Show summary of tree
*/

SET @SQL = 'SELECT * FROM ' + @##HorizTree + ' ORDER BY NodeId'
EXEC dbo.sp_executesql @SQL

-- Show XML of tree (Optional)
/*
SET @SQL = '
SELECT 
Node AS ''@Node'',
NodeId,
ParentId,
numDescendants,
numChildren,
Height,
Depth
FROM ' +  @##HorizTree + '
FOR XML PATH(''Element''),root(''ROOT'') 
'
EXEC dbo.sp_executesql @SQL
*/

END TRY

/*
Error
*/

BEGIN CATCH
SET @ERROR_MESSAGE = ERROR_MESSAGE() 
SET @ERROR_NUMBER = ERROR_NUMBER()
RETURN @ERROR_NUMBER
END CATCH

/*
Normal return 
*/

SET @ERROR_MESSAGE = ''
SET @ERROR_NUMBER = 0

IF @DEBUG = 0	-- If not debugging, drop global temporary tables created by proc
	BEGIN
	SET @SQL = ''
	SELECT @SQL = ISNULL(@SQL+';', '') +'DROP TABLE ' + name
	FROM tempdb..sysobjects
	WHERE NAME LIKE '##%' AND RIGHT(name,LEN(@Suffix )) =  @Suffix
	SET @SQL = RIGHT(@SQL,LEN(@SQL) -1)
	EXEC (@SQL)
	SET @ERROR_MESSAGE = 'Re-run with @DEBUG = 1 to retain global temporary tables'
	PRINT 'Re-run with @DEBUG = 1 to retain global temporary tables'
	END
ELSE
	BEGIN		-- If debugging, display current @Suffix
	SET @ERROR_MESSAGE = 'SELECT * FROM ' + @##HorizTree
	PRINT 'SELECT * FROM ' + @##HorizTree
	PRINT 'Drop this tab in SSMS to drop global temporary tables created by this session'
	END

RETURN @ERROR_NUMBER

END
GO


